//# 请求工具类
import config from "../config/config.js";
import { handleUnauthorized } from "../utils/index.js";

/**
 * 请求工具类
 */
class Request {
  /**
   * 构造请求头
   * @param {Object} headers 自定义请求头
   * @returns {Object} 合并后的请求头
   */
  static getHeaders(headers = {}) {
    // 从 chrome.storage 获取 token
    const token = localStorage.getItem(config.STORAGE_KEYS.TOKEN);

    return {
      "Content-Type": "application/json",
      ...(token ? { Authorization: `Bearer ${token}` } : {}),
      ...headers,
    };
  }

  /**
   * 处理响应
   * @param {Response} response Fetch API 的响应对象
   */
  static async handleResponse(response) {
    const data = await response.json();

    if (!response.ok) {
      // 处理特定错误码
      switch (response.status) {
        case 401:
          // token过期或未授权
          handleUnauthorized();
          throw new Error("登录已过期，请重新登录");
        case 403:
          throw new Error("没有权限访问");
        case 404:
          throw new Error("请求的资源不存在");
        // case 500:
        //   throw new Error("服务器错误");
        default:
          throw new Error(data.message || "请求失败");
      }
    }

    return data;
  }

  /**
   * 发送请求
   * @param {string} url 请求地址
   * @param {Object} options 请求配置
   */
  static async fetch(url, options = {}) {
    const baseUrl = config.getBaseUrl();
    const fullUrl = `${baseUrl}${url}`;

    const defaultOptions = {
      headers: this.getHeaders(options.headers),
      timeout: config.TIMEOUT,
    };

    // 合并配置
    const fetchOptions = {
      ...defaultOptions,
      ...options,
      headers: {
        ...defaultOptions.headers,
        ...(options.headers || {}),
      },
    };

    try {
      // 创建 AbortController 用于请求超时控制
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), config.TIMEOUT);

      const response = await fetch(fullUrl, {
        ...fetchOptions,
        signal: controller.signal,
      });

      clearTimeout(timeoutId);
      return await this.handleResponse(response);
    } catch (error) {
      if (error.name === "AbortError") {
        throw new Error("请求超时");
      }
      throw error;
    }
  }

  // GET 请求
  static get(url, options = {}) {
    return this.fetch(url, { ...options, method: "GET" });
  }

  // POST 请求
  static post(url, data, options = {}) {
    return this.fetch(url, {
      ...options,
      method: "POST",
      body: JSON.stringify(data),
    });
  }

  // PUT 请求
  static put(url, data, options = {}) {
    return this.fetch(url, {
      ...options,
      method: "PUT",
      body: JSON.stringify(data),
    });
  }

  // DELETE 请求
  static delete(url, data, options = {}) {
    return this.fetch(url, {
      ...options,
      method: "DELETE",
      body: JSON.stringify(data),
    });
  }
}

export default Request;
