import Request from "../utils/request.js";
import { showMessage } from "../utils/index.js";

/**
 * 限时销售设置相关服务
 */
class TimeSaleSettingService {
  /**
   * 创建优惠规则
   * @param {Object} params - 请求参数
   * @param {number} params.userSettingId - 店铺设置ID
   * @param {number} params.totalLimit - 总数量限制
   * @param {number} params.perPersonLimit - 每人限购数量
   * @param {number} params.durationDays - 设置时长（天）
   * @param {Array<Object>} params.priceRules - 价格规则数组（最多30条）
   * @returns {Promise<Object>} 创建结果
   */
  static async createSaleRule(params) {
    try {
      const response = await Request.post("/discount/rules", params);

      if (response.success) {
        return response.data;
      } else {
        showMessage(response.message || "创建优惠规则失败", "error");
        throw new Error(response.message || "创建优惠规则失败");
      }
    } catch (error) {
      showMessage(error || "创建优惠规则失败", "error");
      console.error("创建优惠规则失败:", error);
      throw error;
    }
  }

  /**
   * 更新优惠规则
   * @param {number} userSettingId - 要更新的规则ID（路径参数）
   * @param {Object} params - 请求参数
   * @param {number} params.totalLimit - 总数量限制
   * @param {number} params.perPersonLimit - 每人限购数量
   * @param {number} params.durationDays - 设置时长（天）
   * @param {Array<Object>} params.priceRules - 价格规则数组
   * @returns {Promise<Object>} 更新结果
   */
  static async updateSaleRule(userSettingId, params) {
    try {
      const response = await Request.put(
        `/discount/rules/${userSettingId}`,
        params
      );

      if (response.success) {
        return response.data;
      } else {
        showMessage(response.message || "更新优惠规则失败", "error");
        throw new Error(response.message || "更新优惠规则失败");
      }
    } catch (error) {
      showMessage(error || "更新优惠规则失败", "error");
      console.error("更新优惠规则失败:", error);
      throw error;
    }
  }

  /**
   * 获取优惠规则
   * @returns {Promise<Object>} 获取结果
   */
  static async getSaleRule() {
    try {
      const response = await Request.get(`/discount/rules`);

      if (response.success) {
        return response.data;
      } else {
        showMessage(response.message || "获取优惠规则失败", "error");
        throw new Error(response.message || "获取优惠规则失败");
      }
    } catch (error) {
      showMessage(error || "获取优惠规则失败", "error");
      console.error("获取优惠规则失败:", error);
      throw error;
    }
  }

  /**
   * 重置用户的优惠规则设置
   * @returns {Promise<Object>} 重置结果
   */
  static async resetSaleRule() {
    try {
      const response = await Request.delete("/discount/rules");

      if (response.success) {
        return response.data;
      } else {
        showMessage(response.message || "重置优惠规则失败", "error");
        throw new Error(response.message || "重置优惠规则失败");
      }
    } catch (error) {
      showMessage(error || "重置优惠规则失败", "error");
      console.error("重置优惠规则失败:", error);
      throw error;
    }
  }

  /**
   * 开始同步任务
   * @param {string} platformShopId - 平台店铺ID
   * @returns {Promise<Object>} 同步任务创建结果
   */
  static async startSyncTask(platformShopId) {
    try {
      const response = await Request.post("/shop/sync/start", {
        platformShopId,
      });

      if (response.success) {
        return response;
      } else {
        showMessage(response.message || "开始同步任务失败", "error");
        throw new Error(response.message || "开始同步任务失败");
      }
    } catch (error) {
      showMessage(error || "开始同步任务失败", "error");
      console.error("开始同步任务失败:", error);
      throw error;
    }
  }

  /**
   * 查询任务状态
   * @param {number} taskId - 任务ID
   * @returns {Promise<Object>} 任务状态查询结果
   */
  static async getTaskStatus(taskId) {
    try {
      const response = await Request.get(`/shop/sync/status?taskId=${taskId}`);

      if (response.success) {
        return response.data;
      } else {
        showMessage(response.message || "查询任务状态失败", "error");
        throw new Error(response.message || "查询任务状态失败");
      }
    } catch (error) {
      showMessage(error || "查询任务状态失败", "error");
      console.error("查询任务状态失败:", error);
      throw error;
    }
  }

  /**
   * 获取任务结果
   * @param {number} taskId - 任务ID
   * @returns {Promise<Object>} 任务结果
   */
  static async getTaskResult(taskId) {
    try {
      const response = await Request.get(`/shop/sync/result?taskId=${taskId}`);

      if (response.success) {
        return response.data;
      } else {
        showMessage(response.message || "获取任务结果失败", "error");
        throw new Error(response.message || "获取任务结果失败");
      }
    } catch (error) {
      showMessage(error || "获取任务结果失败", "error");
      console.error("获取任务结果失败:", error);
      throw error;
    }
  }

  /**
   * 获取店铺商品列表、优惠规则和设置状态统计
   * @param {string} platformShopId - 平台店铺ID
   * @returns {Promise<Object>} 获取结果
   */
  static async getShopDiscountPrepare(platformShopId) {
    try {
      const response = await Request.get(
        `/shop/discount/prepare?platformShopId=${platformShopId}`
      );

      if (response.success) {
        return response;
      } else {
        showMessage(
          response.message || "获取店铺商品列表、优惠规则和设置状态统计失败",
          "error"
        );
        throw new Error(
          response.message || "获取店铺商品列表、优惠规则和设置状态统计失败"
        );
      }
    } catch (error) {
      showMessage(
        error || "获取店铺商品列表、优惠规则和设置状态统计失败",
        "error"
      );
      console.error("获取店铺商品列表、优惠规则和设置状态统计失败:", error);
      throw error;
    }
  }

  /**
   * 更新单个商品的优惠设置状态
   * @param {Object} params - 请求参数
   * @param {string} params.platformShopId - 平台店铺ID
   * @param {string} params.productCode - 商品代码
   * @param {boolean} params.success - 设置结果
   * @param {string} params.message - 结果描述
   * @returns {Promise<Object>} 更新结果
   */
  static async updateShopDiscountCallback(params) {
    try {
      const response = await Request.post("/shop/discount/callback", params);

      if (response.success) {
        return response.data;
      } else {
        showMessage(
          response.message || "更新单个商品的优惠设置状态失败",
          "error"
        );
        throw new Error(response.message || "更新单个商品的优惠设置状态失败");
      }
    } catch (error) {
      showMessage(error || "更新单个商品的优惠设置状态失败", "error");
      console.error("更新单个商品的优惠设置状态失败:", error);
      throw error;
    }
  }

  /**
   * 检查同步商品任务状态
   * @param {Object} params - 请求参数
   * @param {string} [params.platformUserId] - 平台用户ID（当不传taskId时必填）
   * @param {string} [params.taskId] - 任务ID，用于查询特定任务的状态
   * @returns {Promise<Object>} 检查结果
   */
  static async checkSyncTask(params) {
    try {
      let url = "/products/sync/check";
      const queryParams = [];

      if (params?.platformUserId) {
        queryParams.push(`platform_user_id=${params.platformUserId}`);
      }

      if (params?.taskId) {
        queryParams.push(`task_id=${params.taskId}`);
      }

      if (queryParams.length > 0) {
        url += `?${queryParams.join("&")}`;
      }

      const response = await Request.get(url);

      if (response.success) {
        return response;
      } else {
        showMessage(response.message || "检查同步商品任务状态失败", "error");
        throw new Error(response.message || "检查同步商品任务状态失败");
      }
    } catch (error) {
      showMessage(error || "批量添加商品失败", "error");
      console.error("检查同步商品任务状态失败:", error);
      throw error;
    }
  }

  /**
   * 批量添加商品
   * @param {Object} params - 请求参数
   * @param {string} params.platform_user_id - 平台用户ID
   * @param {Array<string>} params.shop_Code - 商品代码数组
   * @returns {Promise<Object>} 批量添加结果
   */
  static async batchAddProducts(params) {
    try {
      // 验证参数
      if (
        !params.platform_user_id ||
        !params.shop_Code ||
        !Array.isArray(params.shop_Code)
      ) {
        throw new Error("请提供平台用户ID和商品代码列表");
      }

      if (params.shop_Code.length === 0) {
        throw new Error("商品代码列表不能为空");
      }

      if (params.shop_Code.length > 500) {
        throw new Error("单次处理商品数量不能超过500条");
      }

      const response = await Request.post("/products/batch", params);

      if (response.success) {
        return response.data;
      } else {
        showMessage(response.message || "批量添加商品失败", "error");
        throw new Error(response.message || "批量添加商品失败");
      }
    } catch (error) {
      showMessage(error || "批量添加商品失败", "error");
      console.error("批量添加商品失败:", error);
      throw error;
    }
  }

  /**
   * 完成同步任务
   * @param {Object} params - 请求参数
   * @param {number} params.task_id - 同步任务ID
   * @param {string} params.platform_user_id - 平台用户ID
   * @returns {Promise<Object>} 完成同步任务的结果
   */
  static async completeSyncTask(params) {
    try {
      // 验证参数
      if (!params.task_id || !params.platform_user_id) {
        throw new Error("请提供正确的任务ID和平台用户ID");
      }

      const response = await Request.post("/products/sync/complete", params);

      if (response.success) {
        return response.data;
      } else {
        showMessage(response.message || "完成同步任务失败", "error");
        throw new Error(response.message || "完成同步任务失败");
      }
    } catch (error) {
      showMessage(error || "完成同步任务失败", "error");
      console.error("完成同步任务失败:", error);
      throw error;
    }
  }
}

export default TimeSaleSettingService;
