import Request from "../utils/request.js";
import config from "../config/config.js";
import { showMessage } from "../utils/index.js";

/**
 * 认证相关服务
 */
class AuthService {
  /**
   * 用户注册
   * @param {Object} params - 注册参数
   * @param {string} params.phone - 手机号
   * @param {string} params.password - 密码
   * @param {string} params.countryCode - 国家代码 (例如: CN, JP, KR)
   * @param {string} [params.name] - 用户名 (可选)
   * @returns {Promise<Object>} 注册结果
   */
  static async register(params) {
    try {
      const response = await Request.post("/auth/register", params);

      if (response.success) {
        return response.data;
      } else {
        showMessage(response.message || "注册失败", "error");
        throw new Error(response.message || "注册失败");
      }
    } catch (error) {
      showMessage(error || "注册失败", "error");
      console.error("注册失败:", error);
      throw error;
    }
  }

  /**
   * 用户登录
   * @param {Object} params 登录参数
   * @param {string} params.username 用户名
   * @param {string} params.password 密码
   * @returns {Promise<Object>} 登录结果
   */
  static async login(params) {
    try {
      const response = await Request.post("/auth/login", params);

      // 确保登录成功
      if (response.success && response.data) {
        // 保存token和用户信息
        localStorage.setItem(config.STORAGE_KEYS.TOKEN, response.data.token);
        localStorage.setItem(
          config.STORAGE_KEYS.USER_INFO,
          JSON.stringify(response.data.user)
        );

        return response.data;
      } else {
        showMessage(response.message || "登录失败", "error");
        throw new Error(response.message || "登录失败");
      }
    } catch (error) {
      showMessage(error || "登录失败", "error");
      console.error("Login failed:", error);
      throw error;
    }
  }

  /**
   * 退出登录
   */
  static async logout() {
    try {
      await Request.post("/auth/logout");
    } finally {
      // 无论请求是否成功，都清除本地存储的认证信息
      localStorage.removeItem(config.STORAGE_KEYS.TOKEN);
      localStorage.removeItem(config.STORAGE_KEYS.USER_INFO);
      localStorage.removeItem(config.STORAGE_KEYS.REMEMBER_PASSWORD);
    }
  }

  /**
   * 获取当前登录用户信息
   * @returns {Promise<Object>} 用户信息
   */
  static async getCurrentUser() {
    try {
      return await Request.get("/auth/user");
    } catch (error) {
      console.error("Get current user failed:", error);
      throw error;
    }
  }

  /**
   * 检查是否已登录
   * @returns {boolean}
   */
  static isLoggedIn() {
    return !!localStorage.getItem(config.STORAGE_KEYS.TOKEN);
  }

  /**
   * 获取当前登录用户信息
   * @returns {string|null} 用户信息
   */
  static getUserInfo() {
    const userInfo = localStorage.getItem(config.STORAGE_KEYS.USER_INFO);
    return userInfo;
  }

  /**
   * 保存记住的密码
   * @param {Object} credentials 用户凭证
   */
  static saveRememberPassword(credentials) {
    if (!credentials.username || !credentials.password) return;

    const data = {
      username: credentials.username,
      password: credentials.password,
      timestamp: new Date().getTime(),
    };

    localStorage.setItem(
      config.STORAGE_KEYS.REMEMBER_PASSWORD,
      JSON.stringify(data)
    );
  }

  /**
   * 获取记住的密码
   * @returns {Object|null} 用户凭证
   */
  static getRememberPassword() {
    const savedData = localStorage.getItem(
      config.STORAGE_KEYS.REMEMBER_PASSWORD
    );
    if (!savedData) return null;

    try {
      return JSON.parse(savedData);
    } catch (error) {
      console.error("Parse remembered password failed:", error);
      return null;
    }
  }

  /**
   * 清除记住的密码
   */
  static clearRememberPassword() {
    localStorage.removeItem(config.STORAGE_KEYS.REMEMBER_PASSWORD);
  }
}

export default AuthService;
