export class Pagination {
    constructor(container, options = {}) {
        this.container = container;
        this.options = {
            total: 0,           // 总条数
            pageSize: 10,       // 每页显示条数
            current: 1,         // 当前页码
            showPageNumbers: 5, // 显示的页码数量
            onChange: null,     // 页码改变的回调函数
            ...options
        };
        
        this.init();
    }
    
    init() {
        this.render();
        this.bindEvents();
    }
    
    // 计算总页数
    get totalPages() {
        return Math.ceil(this.options.total / this.options.pageSize);
    }
    
    // 渲染分页组件
    render() {
        const { current } = this.options;
        const total = this.totalPages;
        
        // 更新DOM，使用更简洁的结构
        this.container.innerHTML = `
            <div class="pagination">
                <button class="page-btn first-page" ${current === 1 ? 'disabled' : ''} title="首页">
                    <img src="../../../assets/pagination/first-page.png" alt="首页">
                </button>
                <button class="page-btn prev-page" ${current === 1 ? 'disabled' : ''} title="上一页">
                    <img src="../../../assets/pagination/prev-page.png" alt="上一页">
                </button>
                <span class="current-page">${current}</span>
                <button class="page-btn next-page" ${current === total ? 'disabled' : ''} title="下一页">
                    <img src="../../../assets/pagination/next-page.png" alt="下一页">
                </button>
                <button class="page-btn last-page" ${current === total ? 'disabled' : ''} title="末页">
                    <img src="../../../assets/pagination/last-page.png" alt="末页">
                </button>
            </div>
        `;
    }
    
    // 绑定事件
    bindEvents() {
        this.container.addEventListener('click', (e) => {
            const target = e.target;
            
            if (target.classList.contains('first-page')) {
                this.goToPage(1);
            } else if (target.classList.contains('last-page')) {
                this.goToPage(this.totalPages);
            } else if (target.classList.contains('prev-page')) {
                this.goToPage(this.options.current - 1);
            } else if (target.classList.contains('next-page')) {
                this.goToPage(this.options.current + 1);
            } else if (target.classList.contains('page-number')) {
                const page = parseInt(target.dataset.page);
                this.goToPage(page);
            }
        });
    }
    
    // 跳转到指定页
    goToPage(page) {
        if (page === this.options.current) return;
        if (page < 1 || page > this.totalPages) return;
        
        this.options.current = page;
        this.render();
        
        if (typeof this.options.onChange === 'function') {
            this.options.onChange(page);
        }
    }
    
    // 更新配置
    update(options) {
        this.options = { ...this.options, ...options };
        this.render();
    }
} 