import { showMessage } from "../../../utils/index.js";

export async function initShopDetail(pageManager) {
  // 获取HTML内容
  const response = await fetch(
    chrome.runtime.getURL("popup/components/shopDetail/shopDetail.html")
  );
  const html = await response.text();

  // 插入到容器
  const container = document.getElementById("shopDetailComponentContainer");
  container.innerHTML = html;

  // 初始化店铺数据
  const shopListContainer = container.querySelector(".shop-detail-content");
  // 分页相关状态
  let currentPage = 1;
  const pageSize = 3;
  let totalPages = 1;

  async function loadShopData() {
    try {
      const { ShopService } = await import(
        chrome.runtime.getURL("services/index.js")
      );
      const shopData = await ShopService.getShopList({
        pageNumber: currentPage,
        pageSize: pageSize,
      });
      renderShopList(shopData.shops);
      totalPages = Math.ceil(shopData.total / pageSize);
      updatePaginationState();
    } catch (error) {
      console.error("获取店铺数据失败:", error);
      shopListContainer.innerHTML = `<div class="error-message">数据加载失败，请稍后重试</div>`;
    }
  }

  function renderShopList(shops) {
    const tbody = shopListContainer.querySelector("tbody");
    tbody.innerHTML = shops
      .map(
        (shop) => `
            <tr>
                <td>${shop.platformUserId}</td>
                <td>${shop.shopUrl}</td>
                <td class="action-column hidden">
                    <button class="delete-shop-btn">
                        <img src="../assets/mainimages/trash.png" alt="删除">
                    </button>
                </td>
            </tr>
        `
      )
      .join("");
  }

  // 初始化加载数据
  const initialization = () => {
    loadShopData();
  };

  // 获取元素并绑定事件
  const shopDetail = container.querySelector(".shop-detail-container");
  const backButton = shopDetail.querySelector(".back-button");
  const deleteButton = shopDetail.querySelector(".delete-button");
  const addShopBtn = shopDetail.querySelector(".add-shop-btn");

  // 获取模态框相关元素
  const modal = shopDetail.querySelector(".add-shop-modal");
  const cancelBtn = modal.querySelector(".cancel-btn");
  const submitBtn = modal.querySelector(".submit-btn");
  const shopUrlInput = modal.querySelector(".shop-url");
  const shopIdInput = modal.querySelector(".shop-id");

  // 获取确认删除模态框相关元素
  const confirmDeleteModal = shopDetail.querySelector(".confirm-delete-modal");
  const confirmDeleteBtn = confirmDeleteModal.querySelector(".confirm-btn");
  const cancelDeleteBtn = confirmDeleteModal.querySelector(".cancel-btn");
  let currentDeleteBtn = null; // 存储当前要删除的按钮

  // 返回按钮点击事件
  backButton.addEventListener("click", () => {
    shopDetail.classList.remove("active");
    pageManager.saveCurrentPage("home"); // 返回时保存首页状态
  });

  // 删除按钮点击事件
  deleteButton.addEventListener("click", () => {
    const actionColumns = shopDetail.querySelectorAll(".action-column");
    actionColumns.forEach((column) => {
      column.classList.remove("hidden");
    });
  });

  // 显示模态框
  addShopBtn.addEventListener("click", () => {
    modal.style.display = "block";
    // 清空输入框
    shopUrlInput.value = "";
    shopIdInput.value = "";
  });

  // 关闭模态框
  const closeModal = () => {
    modal.style.display = "none";
  };

  // 取消按钮关闭模态框
  cancelBtn.addEventListener("click", closeModal);

  // 点击遮罩层关闭模态框
  modal.addEventListener("click", (e) => {
    if (e.target === modal) {
      closeModal();
    }
  });

  // 提交按钮处理
  submitBtn.addEventListener("click", async () => {
    const shopId = shopIdInput.value.trim();
    const shopUrl = shopUrlInput.value.trim();

    if (!shopId || !shopUrl) {
      showMessage("请填写完整信息", "error");
      return;
    }

    try {
      const { ShopService } = await import(
        chrome.runtime.getURL("services/index.js")
      );
      await ShopService.bindShop({
        platformUserId: shopId,
        shopUrl: shopUrl,
      });

      showMessage("店铺绑定成功", "success");
      closeModal();
      await loadShopData(); // 重新加载数据刷新列表
    } catch (error) {
      const message = error.message.includes("已存在")
        ? "您输入的平台用户ID已存在，请检查后重新输入"
        : error.message || "绑定失败";
      showMessage(message, "error");
    }
  });

  // 关闭确认删除模态框
  const closeConfirmModal = () => {
    confirmDeleteModal.style.display = "none";
  };

  // 删除店铺按钮点击事件
  shopDetail.addEventListener("click", (e) => {
    const deleteBtn = e.target.closest(".delete-shop-btn");
    if (deleteBtn) {
      currentDeleteBtn = deleteBtn;
      confirmDeleteModal.style.display = "block";
    }
  });

  // 取消删除
  cancelDeleteBtn.addEventListener("click", closeConfirmModal);

  // 点击遮罩层关闭
  confirmDeleteModal.addEventListener("click", (e) => {
    if (e.target === confirmDeleteModal) {
      closeConfirmModal();
    }
  });

  // 确认删除
  confirmDeleteBtn.addEventListener("click", async () => {
    if (currentDeleteBtn) {
      const row = currentDeleteBtn.closest("tr");
      const [platformUserId, shopUrl] = Array.from(
        row.querySelectorAll("td")
      ).map((td) => td.textContent);

      try {
        const { ShopService } = await import(
          chrome.runtime.getURL("services/index.js")
        );
        await ShopService.deleteShop({ platformUserId, shopUrl });

        // 移除表格行
        row.remove();
        // 显示成功提示
        showMessage("店铺解绑成功", "success");
        // 重新加载数据保持同步
        await loadShopData();
      } catch (error) {
        console.error("删除失败:", error);
        showMessage(error.message || "店铺解绑失败", "error");
      }
    }
    closeConfirmModal();
  });

  // 分页按钮事件处理
  function goFirstPage() {
    currentPage = 1;
    loadShopData();
  }

  function goPreviousPage() {
    if (currentPage > 1) {
      currentPage--;
      loadShopData();
    }
  }

  function goNextPage() {
    if (currentPage < totalPages) {
      currentPage++;
      loadShopData();
    }
  }

  function goLastPage() {
    currentPage = totalPages;
    loadShopData();
  }

  // 更新分页按钮状态
  function updatePaginationState() {
    const firstBtn = shopDetail.querySelector(".first-page");
    const prevBtn = shopDetail.querySelector(".prev-page");
    const nextBtn = shopDetail.querySelector(".next-page");
    const lastBtn = shopDetail.querySelector(".last-page");

    firstBtn.disabled = currentPage === 1;
    prevBtn.disabled = currentPage === 1;
    nextBtn.disabled = currentPage === totalPages;
    lastBtn.disabled = currentPage === totalPages;
  }

  return {
    show() {
      shopDetail.classList.add("active");
      initialization();

      // 绑定分页按钮事件
      shopDetail
        .querySelector(".first-page")
        .addEventListener("click", goFirstPage);
      shopDetail
        .querySelector(".prev-page")
        .addEventListener("click", goPreviousPage);
      shopDetail
        .querySelector(".next-page")
        .addEventListener("click", goNextPage);
      shopDetail
        .querySelector(".last-page")
        .addEventListener("click", goLastPage);
    },
    hide() {
      shopDetail.classList.remove("active");
      pageManager.saveCurrentPage("home");
    },
  };
}
