import { AuthService } from "../../../services/index.js";

export class RegisterManager {
  static instance = null;
  registerContainer = null;

  static getInstance() {
    if (!RegisterManager.instance) {
      RegisterManager.instance = new RegisterManager();
    }
    return RegisterManager.instance;
  }

  constructor() {
    console.log("RegisterManager constructor called");
    this.init();
  }

  /**
   * 初始化注册页面
   */
  async init() {
    console.log("Initializing register page");
    try {
      // 加载HTML模板
      await this.loadRegisterTemplate();
      // 初始化事件监听
      this.initEventListeners();
    } catch (error) {
      console.error("Register initialization failed:", error);
    }
  }

  /**
   * 加载注册页面模板
   */
  async loadRegisterTemplate() {
    console.log("Loading register template");
    try {
      const templateUrl = chrome.runtime.getURL(
        "popup/components/register/register.html"
      );
      const response = await fetch(templateUrl);
      const html = await response.text();
      // 直接插入到 body 末尾，而不是容器中
      document.body.insertAdjacentHTML("beforeend", html);

      // 添加样式
      const link = document.createElement("link");
      link.rel = "stylesheet";
      link.href = chrome.runtime.getURL("popup/components/register/register.css");
      document.head.appendChild(link);

      // 加载容器
      this.registerContainer = document.getElementById("registerContainer");
    } catch (error) {
      console.error("Failed to load register template:", error);
    }
  }

  /**
   * 初始化事件监听
   */
  initEventListeners() {
    if (!this.registerContainer) return;
    // 点击注册按钮
    const registerBtn = document.getElementById("registerButton");
    if (registerBtn) {
      registerBtn.addEventListener("click", () => {
        console.log("registerBtn button clicked");
        this.showLoginModal();
      });
    }

    // 返回按钮点击事件
    const backButton = this.registerContainer.querySelector("#registerBackButton");
    if (backButton) {
      backButton.addEventListener("click", () => {
        this.hideRegisterPage();
      });
    }

    // 清空按钮点击事件
    const clearButton = this.registerContainer.querySelector("#registerClearButton");
    if (clearButton) {
      clearButton.addEventListener("click", () => {
        this.clearForm();
      });
    }

    // 密码显示切换
    const togglePasswordButtons = this.registerContainer.querySelectorAll(".toggle-password");
    togglePasswordButtons.forEach(button => {
      button.addEventListener("click", (e) => {
        this.togglePasswordVisibility(e);
      });
    });

    // 国家代码变更事件
    const countryCodeSelect = this.registerContainer.querySelector("#countryCode");
    if (countryCodeSelect) {
      countryCodeSelect.addEventListener("change", () => {
        this.updatePhoneFormatHint();
      });
      // 初始化手机号格式提示
      this.updatePhoneFormatHint();
    }

    // 表单提交事件
    const registerForm = this.registerContainer.querySelector("#registerForm");
    if (registerForm) {
      registerForm.addEventListener("submit", (e) => {
        e.preventDefault();
        this.handleRegister();
      });
    }

    // 用户协议和隐私政策链接
    const termsLink = this.registerContainer.querySelector("#termsLink");
    const privacyLink = this.registerContainer.querySelector("#privacyLink");
    
    if (termsLink) {
      termsLink.addEventListener("click", (e) => {
        e.preventDefault();
        // 打开用户协议页面
        chrome.tabs.create({ url: "https://www.qoo10bi.com/terms" });
      });
    }
    
    if (privacyLink) {
      privacyLink.addEventListener("click", (e) => {
        e.preventDefault();
        // 打开隐私政策页面
        chrome.tabs.create({ url: "https://www.qoo10bi.com/privacy" });
      });
    }
  }

  /**
   * 更新手机号格式提示
   */
  updatePhoneFormatHint() {
    const countryCode = document.getElementById("countryCode").value;
    const hintElement = this.registerContainer.querySelector(".phone-format-hint");
    
    if (hintElement) {
      switch (countryCode) {
        case "CN":
          hintElement.textContent = "格式示例：13812345678";
          break;
        case "JP":
          hintElement.textContent = "格式示例：07012345678 或 070-1234-5678";
          break;
        case "KR":
          hintElement.textContent = "格式示例：01012345678 或 010-1234-5678";
          break;
        default:
          hintElement.textContent = "请输入有效的手机号码";
      }
    }
  }

  /**
   * 显示注册页面
   */
  showRegisterPage() {
    if (this.registerContainer) {
      this.registerContainer.classList.add("active");
    }
  }

  /**
   * 隐藏注册页面
   */
  hideRegisterPage() {
    if (this.registerContainer) {
      this.registerContainer.classList.remove("active");
    }
  }

  /**
   * 切换密码可见性
   */
  togglePasswordVisibility(event) {
    const button = event.currentTarget;
    const targetId = button.getAttribute("data-target");
    const passwordInput = document.getElementById(targetId);
    const img = button.querySelector("img");

    if (passwordInput && img) {
      if (passwordInput.type === "password") {
        passwordInput.type = "text";
        img.src = "../assets/mainimages/eye-open.png";
      } else {
        passwordInput.type = "password";
        img.src = "../assets/mainimages/eye-close.png";
      }
    }
  }

  /**
   * 清空表单
   */
  clearForm() {
    const form = this.registerContainer.querySelector("#registerForm");
    if (form) {
      form.reset();
      // 清除所有错误提示
      const errorGroups = form.querySelectorAll(".form-group.error");
      errorGroups.forEach(group => {
        group.classList.remove("error");
      });
      // 隐藏错误消息
      const errorMessage = form.querySelector("#registerErrorMessage");
      if (errorMessage) {
        errorMessage.classList.remove("visible");
      }
      // 更新手机号格式提示
      this.updatePhoneFormatHint();
    }
  }

  /**
   * 验证表单
   */
  validateForm() {
    let isValid = true;
    const name = document.getElementById("registerName");
    const countryCode = document.getElementById("countryCode");
    const phone = document.getElementById("registerPhone");
    const password = document.getElementById("registerPassword");
    const confirmPassword = document.getElementById("confirmPassword");
    const agreeTerms = document.getElementById("agreeTerms");

    // 验证姓名
    if (!name.value.trim()) {
      this.showFieldError(name, "请输入姓名");
      isValid = false;
    } else if (name.value.trim().length < 3 || name.value.trim().length > 50) {
      this.showFieldError(name, "姓名长度应为3-50个字符");
      isValid = false;
    } else {
      this.clearFieldError(name);
    }

    // 验证手机号
    if (!phone.value.trim()) {
      this.showFieldError(phone, "请输入手机号");
      isValid = false;
    } else if (!this.isValidPhone(phone.value, countryCode.value)) {
      this.showFieldError(phone, "请输入有效的手机号码");
      isValid = false;
    } else {
      this.clearFieldError(phone);
    }

    // 验证密码
    if (!password.value.trim()) {
      this.showFieldError(password, "请输入密码");
      isValid = false;
    } else if (password.value.length < 6 || password.value.length > 20) {
      this.showFieldError(password, "密码长度应为6-20个字符");
      isValid = false;
    } else {
      this.clearFieldError(password);
    }

    // 验证确认密码
    if (!confirmPassword.value.trim()) {
      this.showFieldError(confirmPassword, "请确认密码");
      isValid = false;
    } else if (confirmPassword.value !== password.value) {
      this.showFieldError(confirmPassword, "两次输入的密码不一致");
      isValid = false;
    } else {
      this.clearFieldError(confirmPassword);
    }

    // 验证用户协议
    if (!agreeTerms.checked) {
      this.showError("请阅读并同意用户协议和隐私政策");
      isValid = false;
    }

    return isValid;
  }

  /**
   * 验证手机号格式
   */
  isValidPhone(phone, countryCode) {
    // 移除所有非数字字符（如连字符、空格等）
    const cleanPhone = phone.replace(/\D/g, '');
    
    switch (countryCode) {
      case "CN": // 中国手机号
        return /^1\d{10}$/.test(cleanPhone);
      case "JP": // 日本手机号
        return /^(070|080|090)\d{8}$/.test(cleanPhone);
      case "KR": // 韩国手机号
        return /^(010|011|016|017|018|019)\d{7,8}$/.test(cleanPhone);
      default:
        return false;
    }
  }

  /**
   * 显示字段错误
   */
  showFieldError(field, message) {
    const formGroup = field.closest(".form-group");
    formGroup.classList.add("error");
    formGroup.querySelector(".error-tip").textContent = message;
  }

  /**
   * 清除字段错误
   */
  clearFieldError(field) {
    const formGroup = field.closest(".form-group");
    formGroup.classList.remove("error");
  }

  /**
   * 显示错误消息
   */
  showError(message) {
    const errorElement = document.getElementById("registerErrorMessage");
    if (errorElement) {
      errorElement.textContent = message;
      errorElement.classList.add("visible");
    }
  }

  /**
   * 隐藏错误消息
   */
  hideError() {
    const errorElement = document.getElementById("registerErrorMessage");
    if (errorElement) {
      errorElement.classList.remove("visible");
    }
  }

  /**
   * 处理注册
   */
  async handleRegister() {
    // 验证表单
    if (!this.validateForm()) {
      return;
    }

    // 获取表单数据
    const name = document.getElementById("registerName").value;
    const countryCodeSelect = document.getElementById("countryCode");
    const countryCode = countryCodeSelect.value;
    const phone = document.getElementById("registerPhone").value.replace(/\D/g, ''); // 移除非数字字符
    const password = document.getElementById("registerPassword").value;

    // 显示加载状态
    const submitBtn = this.registerContainer.querySelector(".register-submit-btn");
    submitBtn.disabled = true;
    submitBtn.textContent = "注册中...";
    this.hideError();

    try {
      // 构建请求数据
      const requestData = {
        phone,
        password,
        countryCode
      };
      
      // 如果用户填写了姓名，添加到请求数据中
      if (name.trim()) {
        requestData.name = name.trim();
      }
      
      // 发送注册请求
      console.log("注册请求:", requestData);
      const result = await AuthService.register(requestData);
      
      // 显示成功消息
      this.showSuccessMessage("注册成功！请返回登录");
      
      // 3秒后返回主页
      setTimeout(() => {
        this.hideRegisterPage();
      }, 3000);
    } catch (error) {
      // 显示错误信息
      this.showError(error.message || "注册失败，请稍后重试");
      console.error("注册失败:", error);
    } finally {
      // 恢复按钮状态
      submitBtn.disabled = false;
      submitBtn.textContent = "注 册";
    }
  }

  /**
   * 显示成功消息
   */
  showSuccessMessage(message) {
    const errorElement = document.getElementById("registerErrorMessage");
    if (errorElement) {
      errorElement.textContent = message;
      errorElement.style.color = "#52c41a";
      errorElement.style.backgroundColor = "#f6ffed";
      errorElement.classList.add("visible");
    }
  }
}

// 导出初始化函数
document.addEventListener("DOMContentLoaded", () => {
  console.log("DOMContentLoaded fired");
  RegisterManager.getInstance();
});
