/**
 * 充值引导弹框组件
 * 用于在用户需要付费功能但没有额度时引导用户前往充值页面
 */

import { PageStateManager } from "../../utils/PageStateManager.js";
import { initMemberRecharge } from "../memberRecharge/memberRecharge.js";

export class RechargeModal {
  constructor() {
    this.modalContainer = null;
    this.initModal();
  }

  /**
   * 初始化模态框
   */
  initModal() {
    // 创建模态框容器
    this.modalContainer = document.createElement("div");
    this.modalContainer.className = "recharge-modal";
    this.modalContainer.innerHTML = `
      <div class="recharge-modal-content">
        <div class="recharge-modal-header">
          <img src="../assets/mainimages/充值中心.png" alt="充值" class="recharge-icon">
          <h2>会员充值提示</h2>
          <button class="close-btn">
            <img src="../assets/mainimages/close.png" alt="关闭">
          </button>
        </div>
        <div class="recharge-modal-body">
          <p>您当前不是付费用户，且已经没有免费使用额度</p>
          <p>充值后即可继续使用全部功能</p>
          <div class="benefits">
            <div class="benefit-item">
              <span class="benefit-icon">✓</span>
              <span>无限次使用限时特价设置</span>
            </div>
            <div class="benefit-item">
              <span class="benefit-icon">✓</span>
              <span>更多高级功能</span>
            </div>
            <div class="benefit-item">
              <span class="benefit-icon">✓</span>
              <span>优先技术支持</span>
            </div>
          </div>
        </div>
        <div class="recharge-modal-footer">
          <button class="cancel-btn">取消</button>
          <button class="confirm-btn">立即充值</button>
        </div>
      </div>
    `;

    // 添加样式
    const style = document.createElement("style");
    style.textContent = `
      .recharge-modal {
        display: none;
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(0, 0, 0, 0.5);
        z-index: 1000;
        justify-content: center;
        align-items: center;
        animation: fadeIn 0.3s ease-out;
      }
      
      @keyframes fadeIn {
        from { opacity: 0; }
        to { opacity: 1; }
      }
      
      @keyframes slideIn {
        from { transform: translateY(-20px); opacity: 0; }
        to { transform: translateY(0); opacity: 1; }
      }
      
      .recharge-modal-content {
        background-color: #fff;
        border-radius: 12px;
        width: 85%;
        max-width: 400px;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15);
        animation: slideIn 0.3s ease-out;
      }
      
      .recharge-modal-header {
        display: flex;
        align-items: center;
        padding: 15px 20px;
        border-bottom: 1px solid #f0f0f0;
        position: relative;
      }
      
      .recharge-icon {
        width: 24px;
        height: 24px;
        margin-right: 10px;
      }
      
      .recharge-modal-header h2 {
        margin: 0;
        font-size: 18px;
        color: #333;
        font-weight: 600;
      }
      
      .close-btn {
        position: absolute;
        right: 15px;
        top: 15px;
        background: none;
        border: none;
        cursor: pointer;
        padding: 0;
      }
      
      .close-btn img {
        width: 16px;
        height: 16px;
      }
      
      .recharge-modal-body {
        padding: 20px;
        text-align: center;
      }
      
      .recharge-modal-body p {
        margin: 0 0 15px;
        color: #666;
        font-size: 14px;
        line-height: 1.5;
      }
      
      .benefits {
        margin-top: 20px;
        text-align: left;
        background-color: #f8f9ff;
        padding: 15px;
        border-radius: 8px;
      }
      
      .benefit-item {
        display: flex;
        align-items: center;
        margin-bottom: 10px;
      }
      
      .benefit-item:last-child {
        margin-bottom: 0;
      }
      
      .benefit-icon {
        display: inline-flex;
        align-items: center;
        justify-content: center;
        width: 20px;
        height: 20px;
        background-color: #6B7AFF;
        color: white;
        border-radius: 50%;
        margin-right: 10px;
        font-size: 12px;
      }
      
      .recharge-modal-footer {
        display: flex;
        justify-content: space-between;
        padding: 15px 20px 20px;
      }
      
      .cancel-btn, .confirm-btn {
        padding: 10px 0;
        border-radius: 6px;
        font-size: 14px;
        font-weight: 500;
        cursor: pointer;
        width: 48%;
        border: none;
        transition: all 0.2s ease;
      }
      
      .cancel-btn {
        background-color: #f5f5f5;
        color: #666;
      }
      
      .cancel-btn:hover {
        background-color: #e8e8e8;
      }
      
      .confirm-btn {
        background-color: #6B7AFF;
        color: white;
      }
      
      .confirm-btn:hover {
        background-color: #5A68E8;
      }
    `;

    // 添加到DOM
    document.head.appendChild(style);
    document.body.appendChild(this.modalContainer);

    // 绑定事件
    this.bindEvents();
  }

  /**
   * 绑定事件处理
   */
  bindEvents() {
    const closeBtn = this.modalContainer.querySelector(".close-btn");
    const cancelBtn = this.modalContainer.querySelector(".cancel-btn");
    const confirmBtn = this.modalContainer.querySelector(".confirm-btn");

    // 关闭按钮
    closeBtn.addEventListener("click", () => this.hide());

    // 取消按钮
    cancelBtn.addEventListener("click", () => this.hide());

    // 确认按钮 - 跳转到充值页面
    confirmBtn.addEventListener("click", () => {
      this.hide();
      this.navigateToRecharge();
    });

    // 点击模态框外部关闭
    this.modalContainer.addEventListener("click", (e) => {
      if (e.target === this.modalContainer) {
        this.hide();
      }
    });
  }

  /**
   * 显示模态框
   */
  show() {
    this.modalContainer.style.display = "flex";
  }

  /**
   * 隐藏模态框
   */
  hide() {
    this.modalContainer.style.display = "none";
  }

  /**
   * 导航到充值页面
   */
  async navigateToRecharge() {
    // 关闭当前页面组件
    this.closeCurrentPage();
    
    // 使用PageStateManager导航到充值页面
    const pageManager = new PageStateManager();
    pageManager.saveCurrentPage("memberRecharge");
    
    try {
      // 初始化并显示会员充值组件
      const memberRechargeComponent = await initMemberRecharge();
      if (memberRechargeComponent) {
        memberRechargeComponent.show();
      }
    } catch (error) {
      console.error("Error showing member recharge:", error);
    }
  }
  
  /**
   * 关闭当前页面组件
   */
  closeCurrentPage() {
    // 获取当前可能打开的页面组件并关闭
    const timeSaleSettingContainer = document.querySelector(".time-sale-container");
    const shopDetailContainer = document.querySelector(".shop-detail-container");
    
    // 隐藏组件
    if (timeSaleSettingContainer) {
      timeSaleSettingContainer.classList.remove("active");
    }
    
    if (shopDetailContainer) {
      shopDetailContainer.classList.remove("active");
    }
  }

  /**
   * 获取单例实例
   */
  static getInstance() {
    if (!RechargeModal.instance) {
      RechargeModal.instance = new RechargeModal();
    }
    return RechargeModal.instance;
  }
}