import { AuthService } from "../../../services/index.js";

export class LoginManager {
  static instance = null;

  static getInstance() {
    if (!LoginManager.instance) {
      LoginManager.instance = new LoginManager();
    }
    return LoginManager.instance;
  }

  constructor() {
    console.log("LoginManager constructor called");
    this.init();
  }

  async init() {
    try {
      // 先加载HTML模板
      await this.loadLoginTemplate();
      console.log("Template loaded");
      // 初始化事件监听
      this.initEventListeners();
      console.log("Events initialized");
      // 检查登录状态
      this.checkLoginStatus();
      console.log("Login status checked");
    } catch (error) {
      console.error("Initialization failed:", error);
    }
  }

  async loadLoginTemplate() {
    console.log("Loading template...");
    try {
      const templateUrl = chrome.runtime.getURL(
        "popup/components/login/login.html"
      );
      const response = await fetch(templateUrl);
      const html = await response.text();
      // 直接插入到 body 末尾，而不是容器中
      document.body.insertAdjacentHTML("beforeend", html);

      // 添加样式
      const link = document.createElement("link");
      link.rel = "stylesheet";
      link.href = chrome.runtime.getURL("popup/components/login/login.css");
      document.head.appendChild(link);

      console.log("Template and styles loaded");
    } catch (error) {
      console.error("Template loading failed:", error);
    }
  }

  initEventListeners() {
    // 同时使用类选择器和ID选择器来确保能找到按钮
    const loginBtn =
      document.querySelector(".login-btn") ||
      document.getElementById("loginButton");

    if (loginBtn) {
      loginBtn.addEventListener("click", () => {
        console.log("Login button clicked");
        this.showLoginModal();
      });
    }

    // 退出登录按钮
    const logoutBtn = document.getElementById("logoutButton");
    if (logoutBtn) {
      logoutBtn.addEventListener("click", () => {
        this.logout();
      });
    }

    // 关闭按钮点击事件
    const closeBtn = document.getElementById("closeLogin");
    if (closeBtn) {
      closeBtn.addEventListener("click", (e) => {
        console.log("Close button clicked");
        e.preventDefault();
        e.stopPropagation();
        this.hideLoginModal();
      });
    }

    // 点击模态框外部关闭
    // const loginModal = document.getElementById("loginModal");
    // if (loginModal) {
    //   loginModal.addEventListener("click", (e) => {
    //     if (e.target === loginModal) {
    //       this.hideLoginModal();
    //     }
    //   });
    // }

    // 阻止模态框内部点击事件冒泡
    const loginContainer = document.querySelector(".login-container");
    if (loginContainer) {
      loginContainer.addEventListener("click", (e) => {
        e.stopPropagation();
      });
    }

    // 表单提交事件
    document.getElementById("loginForm").addEventListener("submit", (e) => {
      e.preventDefault();
      this.handleLogin();
    });

    // 切换密码显示/隐藏
    const togglePassword = document.getElementById("togglePassword");
    if (togglePassword) {
      togglePassword.addEventListener("click", () => {
        const passwordInput = document.getElementById("password");
        const img = togglePassword.querySelector("img");

        if (passwordInput.type === "password") {
          passwordInput.type = "text";
          img.src = "../assets/mainimages/eye-open.png";
        } else {
          passwordInput.type = "password";
          img.src = "../assets/mainimages/eye-close.png";
        }
      });
    }
  }

  showLoginModal() {
    console.log("Showing login modal");
    const modal = document.getElementById("loginModal");
    console.log("Modal found:", !!modal);
    if (modal) {
      modal.style.display = "block";
      console.log("Modal display style set to:", modal.style.display);

      // 检查计算后的样式
      const computedStyle = window.getComputedStyle(modal);
      console.log("Computed display style:", computedStyle.display);
      console.log("Modal z-index:", computedStyle.zIndex);
      console.log("Modal position:", computedStyle.position);
    }
  }

  hideLoginModal() {
    const modal = document.getElementById("loginModal");
    if (modal) {
      modal.style.display = "none";
    }
  }

  validateForm() {
    let isValid = true;
    const username = document.getElementById("username");
    const password = document.getElementById("password");

    // 验证用户名
    if (!username.value.trim()) {
      this.showFieldError(username, "请输入用户名");
      isValid = false;
    } else {
      this.clearFieldError(username);
    }

    // 验证密码
    if (!password.value.trim()) {
      this.showFieldError(password, "请输入密码");
      isValid = false;
    } else {
      this.clearFieldError(password);
    }

    return isValid;
  }

  showFieldError(field, message) {
    const formGroup = field.closest(".form-group");
    formGroup.classList.add("error");
    formGroup.querySelector(".error-tip").textContent = message;
  }

  clearFieldError(field) {
    const formGroup = field.closest(".form-group");
    formGroup.classList.remove("error");
  }

  async handleLogin() {
    if (!this.validateForm()) {
      return;
    }

    const username = document.getElementById("username").value;
    const password = document.getElementById("password").value;
    const rememberPassword =
      document.getElementById("rememberPassword").checked;

    // 显示加载状态
    const submitBtn = document.querySelector(".login-submit-btn");
    submitBtn.disabled = true;
    submitBtn.textContent = "登录中...";

    try {
      // 使用 AuthService 进行登录
      const loginData = await AuthService.login({ username, password });
      // 如果记住密码，保存凭证
      if (rememberPassword) {
        AuthService.saveRememberPassword({ username, password });
      } else {
        AuthService.clearRememberPassword();
      }

      // 登录成功后更新UI
      this.hideLoginModal();
      this.updateUIAfterLogin(loginData.user);
    } catch (error) {
      // 显示错误信息
      this.showError(error.message || "登录失败，请稍后重试");
    } finally {
      // 恢复按钮状态
      submitBtn.disabled = false;
      submitBtn.textContent = "登 录";
    }
  }

  showError(message) {
    const errorElement = document.getElementById("errorMessage");
    errorElement.textContent = message;
    errorElement.style.display = "block";
  }

  checkLoginStatus() {
    const isLoggedIn = AuthService.isLoggedIn();

    // 如果已经登录，更新UI
    if (isLoggedIn) {
      this.updateUIAfterLogin();
    }

    // 检查是否需要自动填充账号密码
    try {
      const credentials = AuthService.getRememberPassword();
      if (credentials) {
        const usernameInput = document.getElementById("username");
        const passwordInput = document.getElementById("password");
        const rememberCheckbox = document.getElementById("rememberPassword");

        if (usernameInput && passwordInput && rememberCheckbox) {
          usernameInput.value = credentials.username;
          passwordInput.value = credentials.password;
          rememberCheckbox.checked = true;
          console.log("自动填充保存的凭证");
        }
      }
    } catch (error) {
      console.error("获取保存的凭证失败:", error);
    }
  }

  updateUIAfterLogin(userInfo) {
    // 使用ID选择器
    const loginBtn = document.getElementById("loginButton");
    const registerBtn = document.getElementById("registerButton");
    if (loginBtn) {
      loginBtn.style.display = "none";
    }
    if (registerBtn) {
      registerBtn.style.display = "none";
    }

    // 显示隐退出按钮
    const logoutBtn = document.getElementById("logoutButton");
    if (logoutBtn) {
      logoutBtn.style.display = "block";
      // 给头像添加下拉列表
      const avatarDropdown = document.getElementById("avatarDropdown");
      avatarDropdown.classList.toggle("active");
    }

    // 显示用户信息
    const username = userInfo?.name || userInfo?.username;
    // TODO: 这里可以添加显示用户信息的UI元素
    console.log("当前登录用户:", username);
  }

  async logout() {
    try {
      await AuthService.logout();
      // 使用ID选择器
      const loginBtn = document.getElementById("loginButton");
      const registerBtn = document.getElementById("registerButton");
      if (loginBtn) {
        loginBtn.style.display = "block";
      }
      if (registerBtn) {
        registerBtn.style.display = "block";
      }
      // 移除退登按钮
      const avatarDropdown = document.getElementById("avatarDropdown");
      avatarDropdown.classList.remove("active");

      // 如果不需要记住密码，清除保存的凭证
      if (!document.getElementById("rememberPassword")?.checked) {
        AuthService.clearRememberPassword();
      }
    } catch (error) {
      console.error("登出失败:", error);
    }
  }
}

// 确保在 DOM 加载完成后初始化
console.log("Script loaded, waiting for DOMContentLoaded");
document.addEventListener("DOMContentLoaded", () => {
  console.log("DOMContentLoaded fired");
  LoginManager.getInstance();
});
